/*  Beispiel 6.3.2 "7-Segmentanzeige - Adresse finden"
*   
*   Zeigt in der 7-Segmentanzeige die zugeordnete I2C Adresse an.
*   Mögliche Werte: 0x40, 0x44, 0x48, 0x4c
*
*/

#include <Wire.h>

// 8574T standardmaessig verbaut
#define i2cseg7x2alsb1 (0x40>>1) // 1 Bit rechtsschieben: 8 bit >> 7 bit
#define i2cseg7x2amsb1 (0x42>>1) // 1 Bit rechtsschieben: 8 bit >> 7 bit 
#define i2cseg7x2blsb1 (0x44>>1)
#define i2cseg7x2bmsb1 (0x46>>1)
#define i2cseg7x2clsb1 (0x48>>1)
#define i2cseg7x2cmsb1 (0x4A>>1)
#define i2cseg7x2dlsb1 (0x4C>>1)
#define i2cseg7x2dmsb1 (0x4E>>1)


// Aufbau der Segmente, Zuordnung zu den Bits. 
// Codierung der Segemente errechnet sich durch
// Addition der Hex-Zahl der relevanten Segmente
// (siehe folgende ASCII-Graphik)
// ******************************
//          01
//       20    02
//          40
//       10    04
//          08
//                 80
// *****************************
// 0x80 ist fuer den Dezimalpunkt (dp)

// Umrechnungstabelle ASCII -> 7 Segment
// Offset ASCII-Code 32..5F entspricht den Zeichen Space bis Z

const unsigned char siebensegtable[] =
{
// 7-Segment-Codierung, ASCII-Code (Hex), Zeichen (im Klartext) 
   0, // 20 Space
   0x30, // 21 !
   0x22, // 22 "
   0x7f, // 23 #
   0, // 24 $
   0, // 25 %
   0, // 26 &
   0x02, // 27 '
   0x39, // 28 (
   0x0f, // 29 )
   0, // 2A *
   0x7f, // 2B +
   0x04, // 2C ,
   0x40, // 2D -
   0x80, // 2E .
   0x30, // 2F /
   0x3f, // 30 0
   0x06, // 31 1
   0x5b, // 32 2
   0x4f, // 33 3
   0x66, // 34 4
   0x6d, // 35 5
// 0x7c, // 36 6 //alternative Darstellung ohne Segment 01
   0x7D, // 36 6
   0x07, // 37 7
   0x7f, // 38 8
// 0x67, // 39 9 //alternative Darstellung ohne Segment 08
   0x6F, // 39 9
   //
   0, // 3A :
   0, // 3B ;
   0, // 3C <
   0x48, // 3D =
   0, // 3E >
   0, // 3F ?
   0x5c, // 40 @
   0x77, // 41 A
   0x7c, // 42 B
   0x39, // 43 C
   0x5e, // 44 D
   0x79, // 45 E
   0x71, // 46 F
   0x67, // 47 G
   0x76, // 48 H
   0x06, // 49 I
   0x86, // 4A J
   0x74, // 4B K
   0x38, // 4C L
   0x37, // 4D M
   0x54, // 4E N
   0x5c, // 4F O
   0x73, // 50 P
   0xbf, // 51 Q
   0x50, // 52 R
   0x6d, // 53 S
   0x70, // 54 T
   0x3e, // 55 U
   0x1c, // 56 V
   0x9c, // 57 W
   0x24, // 58 X
   0x36, // 59 Y
   0x5b, // 5A Z
   0x39, // 5B [
   0x30, // 5C
   0x0f, // 5D ]
   0x08, // 5E _
   0 // 5F OHNE
};

// Umrechnen ASCII Code in Tabellenindex
unsigned int get_7seg(unsigned char asciicode)
{
    // Umrechnen 0..255 auf
    // 7 seg Tabellenindex
    // Dabei nur Zahlen und Grossbuchstaben
    // 20..5F
    // Rest wird auf diese gemappt
    asciicode = asciicode & 0x7f; // nur 7 Bits 
    if (asciicode < 0x20) return (0); // Sonderzeichen nicht
    if (asciicode >= 0x60) asciicode = asciicode - 0x20; // Kleinbuchstaben
    return((~siebensegtable[asciicode-0x20])&0xff); // Index zurueck
}

// Anzeige eines einzelnen ASCII Zeichen, das als 
// char übergeben wird. Ausgabe ueber den 7 Segmentbrick
// Dazu die Segmentadresse als Parameter uebergeben
// Ohne Dezimalpunkt ausgeben.
void display_seg1x(unsigned char i2cbaseadr, unsigned char ch1)
{
  Wire.beginTransmission(i2cbaseadr); // I2C Adresse
  Wire.write(get_7seg(ch1)); // Tabellenidnex nehmen und dann ausgeben
  Wire.endTransmission(); // Ende I2C
}

// Ausgabe ohne Umrechnung, wenn eigene Zeichen verwendet werden
// sollen. Parameter ist der Bianry Code 
void display_seg1xbin(unsigned char i2cbaseadr, unsigned char ch1)
{ 
  Wire.beginTransmission(i2cbaseadr); // I2C Adresse
  Wire.write(ch1); // Binaercode direkt am Port ausgeben
  Wire.endTransmission(); // Ende I2C
}

// Start
void setup() {
  Wire.begin(); /// I2C Initialisieren
}

void loop() {
// 7-Segmentanzeige mit Treiberbaustein 8574T 
// Alle potentiellen Adressen zur Identifikation ausgeben
// Man kann so sehen welche Adresse man eingestellt hat
  display_seg1x(i2cseg7x2amsb1,'4'); // eigene Adresse
  display_seg1x(i2cseg7x2alsb1,'0'); // ausgeben
  display_seg1x(i2cseg7x2bmsb1,'4'); // sind immer PAARE
  display_seg1x(i2cseg7x2blsb1,'4'); // zwei Befehle fuer einen BRICK
  display_seg1x(i2cseg7x2cmsb1,'4'); 
  display_seg1x(i2cseg7x2clsb1,'8'); // von 0x40 bis 0x4C 
  display_seg1x(i2cseg7x2dmsb1,'4');
  display_seg1x(i2cseg7x2dlsb1,'C');
}

